function [dyn, u] = driverModelBasic(dyn, mission, vehModel, veh, u, driver)
% vehAcc: current acceleration
% vehSpd: current speed
% vehSpd_next: speed at the next timestep

vehSpd = dyn.vehSpd;
vehSpd_next = dyn.vehSpd + dyn.vehAcc * veh.dt;
vehAcc = dyn.vehAcc;

GN = u{1};

%% Determine the driver state
% By default
driver.state = "drive";

%% Behavioural acc. limits
if isfield(veh.ctrl, 'accLimitUp')
    accLimitUp = veh.ctrl.accLimitUp.eval(vehSpd);
end
if isfield(veh.ctrl, 'accLimitLo')
    accLimitLo = veh.ctrl.accLimitLo.eval(vehSpd);
end

if vehAcc > accLimitUp
    vehAcc = accLimitUp;
    vehSpd_next = vehSpd + vehAcc .* veh.dt;
elseif vehAcc < accLimitLo
    vehAcc = accLimitLo;
    vehSpd_next = vehSpd + vehAcc .* veh.dt;
end

%% Try the desired acceleration
GN = max( GN, 1 );
brakeCmd = 0;
u = {GN, brakeCmd};
dyn.vehAcc = vehAcc;
[~, unfeas, prof] = vehModel(u, dyn);

%% Braking
switch veh.vehType
    case "conv"
        if prof.engTrq < prof.engMotTrq
            driver.state = "brake";
            dyn.vehAcc = vehAcc;
            brakeCmd = searchbrakeCmdConv();
        end

    case "bev"
        if prof.emTrq < prof.emMinTrq
            driver.state = "brake";
            dyn.vehAcc = vehAcc;
            brakeCmd = searchbrakeCmdBEV();
        end

end

%% Acceleration limiting
if strcmp(driver.state, "drive") && prof.accUnfeas
    % Search maximum acceleration
    vehAccGrid = -1:0.01:vehAcc; % m/s^2
    try
        dyn.vehAcc = searchAcc(vehAccGrid, "max");
    catch
        dyn.vehAcc = vehAcc;
    end
end

%% Limits based on engine and e-machines speed limits
if vehSpd_next > veh.maxSpd
    vehSpd_next = veh.maxSpd;
    dyn.vehAcc = ( vehSpd_next - vehSpd ) / veh.dt;
end

%% Return new brake command
u{2} = brakeCmd;

%% Helper functions
    function vehAcc = searchAcc(vehAccGrid, minOrMax)
        vehAccGrid = sort(vehAccGrid);

        % Try the provided acceleration grid
        dyn.vehAcc = vehAccGrid;
        [~, unfeas] = vehModel(u, dyn);

        switch minOrMax
            case "min"
                % Index of the strongest deceleration that is feasible
                maxAccIdx = find(unfeas == 0, 1, 'first');
            case "max"
                % Index of the strongest acceleration that is feasible
                maxAccIdx = find(unfeas == 0, 1, 'last');
        end

        if ~isempty(maxAccIdx)
            % Keep the strongest acceleration/deceleration that is feasible and
            % re-evaluate vehSpd_next
            vehAcc = vehAccGrid(maxAccIdx);
        end
    end

    function brakeCmd = searchbrakeCmdConv()
        % Find brakeCmd that gives the desired vehAcc while the engine/EM
        % at maximum motoring
        
        GN = max( GN, 1 );
        brakeCmd = fminbnd(@brakeObjfun, 0, 1);
        brakeCmd = brakeCmd.*1.001; % fminbnd tends to err such that engTrq is slighlty smaller than motTrq 

        function obj = brakeObjfun(x)
            [~, ~, prof] = vehModel({GN, x}, dyn);
            obj = (prof.engTrq - prof.engMotTrq).^2;
        end

    end

    function brakeCmd = searchbrakeCmdBEV()
        % Find brakeCmd that gives the desired vehAcc while the engine/EM
        % at maximum motoring
        
        GN = max( GN, 1 );
        brakeCmd = fminbnd(@brakeObjfun, 0, 1);
        brakeCmd = brakeCmd.*1.001; % fminbnd tends to err such that emTrq is slighlty smaller than minTrq

        function obj = brakeObjfun(x)
            [~, ~, prof] = vehModel({GN, x}, dyn);
            obj = (prof.emTrq - prof.emMinTrq).^2;
        end

    end


end